import { PageType } from '../constants.js';
import { traverse, traverseList } from '../utils/traverse.js';
import AlbumParser from './album_parser.js';
import ArtistParser from './artist_parser.js';
import PlaylistParser from './playlist_parser.js';
import SongParser from './song_parser.js';
export default class Parser {
    static parseDuration(time) {
        if (!time)
            return null;
        const [seconds, minutes, hours] = time
            .split(':')
            .reverse()
            .map((n) => +n);
        return (seconds || 0) + (minutes || 0) * 60 + (hours || 0) * 60 * 60;
    }
    static parseNumber(string) {
        if (string.at(-1).match(/^[A-Z]+$/)) {
            const number = +string.slice(0, -1);
            const multiplier = string.at(-1);
            return ({
                K: number * 1000,
                M: number * 1000 * 1000,
                B: number * 1000 * 1000 * 1000,
                T: number * 1000 * 1000 * 1000 * 1000,
            }[multiplier] || Number.NaN);
        }
        else {
            return +string;
        }
    }
    static parseMixedContent(data) {
        const key = Object.keys(data)[0];
        if (!key)
            throw new Error('Invalid content');
        const result = data[key];
        const musicDescriptionShelfRenderer = traverse(result, 'musicDescriptionShelfRenderer');
        if (musicDescriptionShelfRenderer && !Array.isArray(musicDescriptionShelfRenderer)) {
            return {
                title: traverse(musicDescriptionShelfRenderer, 'header', 'title', 'text'),
                contents: traverseList(musicDescriptionShelfRenderer, 'description', 'runs', 'text'),
            };
        }
        if (!Array.isArray(result.contents)) {
            return null;
        }
        const title = traverse(result, 'header', 'title', 'text');
        const contents = [];
        result.contents.forEach((content) => {
            const musicTwoRowItemRenderer = traverse(content, 'musicTwoRowItemRenderer');
            if (musicTwoRowItemRenderer && !Array.isArray(musicTwoRowItemRenderer)) {
                const pageType = traverse(result, 'navigationEndpoint', 'browseEndpoint', 'browseEndpointContextSupportedConfigs', 'browseEndpointContextMusicConfig', 'pageType');
                const playlistId = traverse(content, 'navigationEndpoint', 'watchPlaylistEndpoint', 'playlistId');
                switch (pageType) {
                    case PageType.MUSIC_PAGE_TYPE_ARTIST:
                        contents.push(ArtistParser.parseSearchResult(content));
                        break;
                    case PageType.MUSIC_PAGE_TYPE_ALBUM:
                        contents.push(AlbumParser.parseSearchResult(content));
                        break;
                    case PageType.MUSIC_PAGE_TYPE_PLAYLIST:
                        contents.push(PlaylistParser.parseSearchResult(content));
                        break;
                    default:
                        if (playlistId) {
                            contents.push(PlaylistParser.parseWatchPlaylist(content));
                        }
                        else {
                            contents.push(SongParser.parseSearchResult(content));
                        }
                }
            }
            else {
                const musicResponsiveListItemRenderer = traverse(content, 'musicResponsiveListItemRenderer');
                if (musicResponsiveListItemRenderer && !Array.isArray(musicResponsiveListItemRenderer)) {
                    contents.push(SongParser.parseSearchResult(musicResponsiveListItemRenderer));
                }
            }
        });
        return { title, contents };
    }
}
//# sourceMappingURL=parser.js.map